#!/usr/bin/env python
# coding: utf-8

# In[ ]:


# ===================================================================================
# SCRIPT DE AUDITORÍA SEO INICIAL - PLANTILLA PARA AGENCIAS
# ===================================================================================
# Preparado por: Gema Calderón Sayoux (gemacalderonsayoux.com)
# Versión: 1.0
#
# INSTRUCCIONES PARA LA AGENCIA:
# 1. Asegúrate de tener Python instalado en tu ordenador.
# 2. Instala las librerías necesarias ejecutando en tu terminal:
#    pip install requests beautifulsoup4 fpdf2 validators python-whois
# 3. Rellena T-O-D-A la sección "CONFIGURACIÓN DE LA AGENCIA" a continuación.
# 4. Ejecuta el script. Se generará un informe en PDF en la misma carpeta.
# ===================================================================================

import requests
from bs4 import BeautifulSoup
from urllib.parse import urljoin, urlparse
from fpdf import FPDF
import validators
import json
import re
from datetime import datetime
import whois
from fpdf.enums import XPos, YPos

# ===================================================================================
#               ¡CONFIGURACIÓN DE LA AGENCIA! RELLENA TODO AQUÍ
# ===================================================================================

# --- 1. DATOS DE TU AGENCIA Y DEL CLIENTE ---
CONFIGURACION_AGENCIA = {
    # El nombre de tu agencia. Aparecerá en el informe.
    "NOMBRE_AGENCIA": "[EL NOMBRE DE TU AGENCIA]",

    # El título que quieres que aparezca en el PDF.
    "TITULO_INFORME": "Informe de Auditoría SEO Inicial",

    # Introduce la URL COMPLETA del sitio web que quieres auditar.
    "URL_A_AUDITAR": "https://www.ejemplocliente.com",

    # Pega aquí tu clave API de Google PageSpeed Insights. Es MUY recomendable.
    # Obtenla gratis aquí: https://developers.google.com/speed/docs/insights/v5/get-started
    "API_KEY_PAGESPEED": "[PEGA AQUÍ TU API KEY DE GOOGLE]"
}

# --- 2. TUS PROPUESTAS DE SOLUCIÓN (PERSONALIZA TU OFERTA DE SERVICIOS) ---
# Edita el texto de cada "solución" para que refleje cómo TU agencia resuelve cada problema.
# Este es el texto que aparecerá en el informe para cada punto analizado.
SOLUCIONES_PROPUESTAS = {
    "title": "[Describe aquí cómo tu agencia soluciona este problema. Ej: 'Analizaremos las palabras clave con mayor potencial y reescribiremos el título para que sea atractivo, conciso (<60 caracteres) y motive el clic.']",
    "description": "[Describe aquí tu solución. Ej: 'Crearemos una meta descripción persuasiva (~155 caracteres) que incluya un llamado a la acción claro y resalte los beneficios de tu negocio.']",
    "h1": "[Describe aquí tu solución. Ej: 'Nos aseguraremos de que la página tenga una única etiqueta H1, que sea descriptiva y contenga la palabra clave principal de forma natural.']",
    "alt_texts": "[Describe aquí tu solución. Ej: 'Optimizaremos todos los textos alternativos para que describan la imagen e incluyan palabras clave relevantes, mejorando el SEO y la accesibilidad web (WAI).']",
    "httpss": "[Describe aquí tu solución. Ej: 'Gestionaremos la correcta instalación y configuración de un certificado SSL (Let's Encrypt o comercial) para asegurar el sitio y cumplir con los estándares de Google.']",
    "canonical": "[Describe aquí tu solución. Ej: 'Implementaremos una etiqueta canónica auto-referenciada en la página principal para consolidar toda la autoridad en una única URL.']",
    "hreflang": "[Describe aquí tu solución si el cliente es multi-idioma. Ej: 'Si el sitio web se dirige a públicos de diferentes idiomas, implementaremos etiquetas hreflang para mejorar la experiencia del usuario y el SEO internacional.']",
    "core_web_vitals": "[Describe aquí tu solución. Ej: 'Realizaremos un análisis técnico profundo para optimizar imágenes, reducir código innecesario y configurar la caché para lograr una puntuación \'Buena\' en todas las métricas.']",
    "social_links": "[Describe aquí tu solución. Ej: 'Integraremos los iconos y enlaces a todos los perfiles sociales activos para fortalecer la imagen de marca y facilitar la interacción con los clientes.']",
    "favicon": "[Describe aquí tu solución. Ej: 'Crearemos y configuraremos un favicon en los formatos correctos (incluyendo para Apple Touch) para mejorar el branding y la visibilidad.']",
    "robots_txt": "[Describe aquí tu solución. Ej: 'Crearemos o revisaremos el archivo robots.txt para asegurar que guía correctamente a los motores de búsqueda, permitiendo el rastreo de todo el contenido relevante.']",
    "sitemap": "[Describe aquí tu solución. Ej: 'Generaremos un sitemap XML dinámico y lo enviaremos a Google Search Console para acelerar la indexación y asegurar que todo el contenido sea visible.']",
    "domain_age": "[Describe aquí tu solución. Ej: 'La antigüedad es un factor de confianza. Potenciaremos esta autoridad con una estrategia de contenidos y backlinks de alta calidad para construir una reputación sólida.']"
}

# --- 3. TEXTO FINAL DE LA PROPUESTA DE VALOR ---
# Este es el texto que aparecerá al final del informe. Úsalo para explicar los próximos pasos y vender tus servicios.
# Usa {nombre_agencia} para insertar automáticamente el nombre de tu agencia.
TEXTO_PROPUESTA_FINAL = """
Este informe automático revela los puntos críticos más inmediatos. Sin embargo, una estrategia SEO y SEM de éxito que genere clientes de forma consistente requiere un enfoque más profundo:

-  **Estudio de Palabras Clave (Keyword Research):** Identificar las búsquedas exactas que realizan sus clientes potenciales y que tienen intención de compra.
-  **Análisis de la Competencia:** Descubrir qué estrategias están usando sus competidores directos para captar clientes y cómo podemos superarlos.
-  **Auditoría de Contenido y SEO On-Page Completo:** Analizar si el contenido actual responde a la intención de búsqueda y optimizar todas las páginas de servicios clave.
-  **Auditoría del Perfil de Enlaces (Backlinks):** Evaluar la autoridad y calidad de los enlaces que apuntan a su web, un factor de ranking fundamental.
-  **Estrategia de SEO Local y Google Business Profile:** Optimizar su presencia para dominar las búsquedas locales y el mapa de Google.

**El objetivo de {nombre_agencia} es claro: solucionar estos problemas técnicos, implementar una estrategia basada en datos y transformar su web en una herramienta constante de captación de clientes.**

Nos gustaría agendar una breve llamada de 15 minutos para explicarle estos hallazgos y cómo nuestro plan de trabajo puede ayudarle a alcanzar sus objetivos de negocio.
"""
# ===================================================================================
#               FIN DE LA CONFIGURACIÓN. NO MODIFICAR EL CÓDIGO DE ABAJO.
# ===================================================================================


USER_AGENT = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36"

class ReportPDF(FPDF):
    def __init__(self, url, agency_name, report_title):
        super().__init__()
        self.report_url = url
        self.agency_name = agency_name
        self.report_title = report_title
        self.set_auto_page_break(auto=True, margin=15)
        self.set_font('Arial', '', 10)

    def header(self):
        self.set_font('Arial', 'B', 16)
        self.cell(0, 10, self.report_title, new_x=XPos.LMARGIN, new_y=YPos.NEXT, align='C')
        self.set_font('Arial', 'I', 10)
        self.cell(0, 8, f'Informe preparado por: {self.agency_name}', new_x=XPos.LMARGIN, new_y=YPos.NEXT, align='C')
        self.set_font('Arial', '', 10)
        self.cell(0, 8, f'URL Analizada: {self.report_url}', new_x=XPos.LMARGIN, new_y=YPos.NEXT, align='C')
        self.ln(10)

    def footer(self):
        self.set_y(-15)
        self.set_font('Arial', '', 8)
        self.cell(0, 10, f'Página {self.page_no()}', align='C')

    def chapter_title(self, title):
        self.set_font('Arial', 'B', 14)
        self.set_fill_color(220, 220, 220)
        self.cell(0, 10, title, new_x=XPos.LMARGIN, new_y=YPos.NEXT, align='L', fill=True)
        self.ln(5)

    def check_item(self, title, result, status, pain_point, solution):
        status_prefixes = {'ok': '[OK]', 'warning': '[AVISO]', 'error': '[ERROR]', 'info': '[INFO]'}
        status_colors = {'ok': (34, 139, 34), 'warning': (255, 140, 0), 'error': (220, 20, 60), 'info': (70, 130, 180)}
        
        self.set_font('Arial', 'B', 11)
        self.set_text_color(*status_colors.get(status, (0, 0, 0)))
        
        safe_title = title.encode('latin-1', 'replace').decode('latin-1')
        prefix = status_prefixes.get(status, '[INFO]')
        self.cell(0, 7, f'{prefix} {safe_title}', new_x=XPos.LMARGIN, new_y=YPos.NEXT)
        self.set_text_color(0, 0, 0)
        
        def write_safe_multicell(font_style, text_prefix, text_content):
            self.set_font('Arial', font_style, 10)
            full_text = f"{text_prefix}: {text_content}"
            safe_text = full_text.encode('latin-1', 'replace').decode('latin-1')
            self.multi_cell(0, 5, safe_text, new_x=XPos.LMARGIN, new_y=YPos.NEXT)

        write_safe_multicell('', "Resultado", result)
        write_safe_multicell('I', "Punto de Dolor", pain_point)
        write_safe_multicell('B', "Nuestra Propuesta de Solución", solution)
        self.ln(6)


class HomepageAuditor:
    def __init__(self, config, solutions, proposal_text):
        self.config = config
        self.solutions = solutions
        self.proposal_text = proposal_text
        self.url = config['URL_A_AUDITAR']
        self.api_key = config['API_KEY_PAGESPEED']
        self.domain = urlparse(self.url).netloc
        self.soup = None
        self.response = None
        self.pdf = ReportPDF(
            self.url,
            self.config['NOMBRE_AGENCIA'],
            self.config['TITULO_INFORME']
        )

    def run(self):
        print(f"Iniciando auditoría para {self.url}...")
        headers = {'User-Agent': USER_AGENT}
        try:
            self.response = requests.get(self.url, headers=headers, timeout=20)
            self.response.raise_for_status()
            self.soup = BeautifulSoup(self.response.content, 'html.parser')
            print("Página principal obtenida con éxito.")
        except requests.RequestException as e:
            print(f"ERROR: No se pudo acceder a la URL. {e}")
            return

        self.generate_report()

    def generate_report(self):
        self.pdf.add_page()
        
        self.pdf.chapter_title("1. Factores Técnicos On-Page")
        self.check_title()
        self.check_description()
        self.check_h1()
        self.check_canonical()
        self.check_hreflang()
        self.check_https()
        
        self.pdf.add_page()
        self.pdf.chapter_title("2. Contenido y Experiencia de Usuario (UX)")
        self.check_alt_texts()
        self.check_core_web_vitals()
        self.check_social_links()
        self.check_favicon()
        
        self.pdf.chapter_title("3. Factores a Nivel de Dominio y Autoridad")
        self.check_robots_txt()
        self.check_sitemap()
        self.check_domain_age()
        
        self.pdf.add_page()
        self.pdf.chapter_title("4. Próximos Pasos y Propuesta de Valor")
        self.add_next_steps()

        filename = f"Auditoria_SEO_{self.domain}.pdf"
        self.pdf.output(filename)
        print("\n" + "="*50)
        print(f"¡ÉXITO! El informe se ha guardado como: {filename}")
        print("="*50)

    def check_title(self):
        title_tag = self.soup.find('title')
        title_text = title_tag.get_text(strip=True) if title_tag else ""
        length = len(title_text)
        status = 'ok' if 50 <= length <= 60 else 'error'
        self.pdf.check_item("Etiqueta <title>", f"'{title_text}' ({length} caracteres).", status,
                            "Un título mal optimizado reduce drásticamente el CTR en Google y no comunica el valor de la página.",
                            self.solutions['title'])

    def check_description(self):
        desc_tag = self.soup.find('meta', attrs={'name': 'description'})
        desc_text = desc_tag['content'].strip() if desc_tag and 'content' in desc_tag.attrs else ""
        length = len(desc_text)
        status = 'ok' if 120 <= length <= 158 else 'error'
        self.pdf.check_item("Meta Descripción", f"Tiene {length} caracteres.", status,
                            "La meta descripción es tu 'anuncio gratuito' en Google. Si falta o es de baja calidad, se pierde la oportunidad de convencer al usuario.",
                            self.solutions['description'])

    def check_h1(self):
        h1s = self.soup.find_all('h1')
        count = len(h1s)
        status = 'ok' if count == 1 else 'error'
        self.pdf.check_item("Encabezado H1", f"Se encontraron {count} etiquetas H1.", status,
                            "El H1 es el titular principal de la página. Su ausencia o duplicidad confunde a Google sobre el tema principal, afectando el ranking.",
                            self.solutions['h1'])

    def check_alt_texts(self):
        images = self.soup.find_all('img')
        missing_alts = len([img for img in images if not img.get('alt', '').strip()])
        status = 'ok' if missing_alts == 0 else ('warning' if missing_alts < 5 else 'error')
        self.pdf.check_item("Atributos ALT en Imágenes", f"{missing_alts} de {len(images)} imágenes no tienen texto alternativo.", status,
                            "Las imágenes sin ALT son invisibles para Google y para usuarios con discapacidad visual. Se pierde posicionamiento en Google Images y accesibilidad.",
                            self.solutions['alt_texts'])

    def check_https(self):
        is_https = self.response.url.startswith('https://')
        status = 'ok' if is_https else 'error'
        self.pdf.check_item("Certificado de Seguridad (HTTPS)", "El sitio utiliza HTTPS." if is_https else "¡El sitio NO utiliza HTTPS!", status,
                            "Un sitio sin HTTPS es marcado como 'No Seguro' por los navegadores, generando desconfianza. Además, es un factor de ranking negativo para Google.",
                            self.solutions['httpss'])
    
    def check_canonical(self):
        canonical = self.soup.find('link', rel='canonical')
        if canonical and canonical.get('href'):
            href = canonical.get('href')
            status = 'ok' if urlparse(href).path == urlparse(self.url).path else 'warning'
            result = f"Encontrada, apunta a: {href}"
        else:
            status = 'error'
            result = "No se encontró etiqueta canónica."
        self.pdf.check_item("Etiqueta Canónica", result, status,
                            "Sin una etiqueta canónica, Google puede indexar múltiples versiones de la misma página (ej. con/sin 'www'), diluyendo la autoridad y causando contenido duplicado.",
                            self.solutions['canonical'])

    def check_hreflang(self):
        hreflangs = self.soup.find_all('link', rel='alternate', hreflang=True)
        if hreflangs:
            count = len(hreflangs)
            status = 'ok'
            result = f"Se encontraron {count} etiquetas hreflang para gestionar múltiples idiomas."
        else:
            status = 'info'
            result = "No se detectaron etiquetas hreflang."
        self.pdf.check_item("Etiquetas Hreflang (Multi-idioma)", result, status,
                            "Una mala configuración de hreflang puede hacer que Google muestre la versión en inglés a un usuario español, o viceversa, arruinando la experiencia.",
                            self.solutions['hreflang'])

    def check_core_web_vitals(self):
        if not self.api_key or "[PEGA AQUÍ TU API KEY DE GOOGLE]" in self.api_key:
            result = "No se ha proporcionado una API Key de Google PageSpeed."
            status = 'warning'
            pain_point = "Sin la API Key no podemos medir la velocidad real que perciben los usuarios, un factor de ranking crítico."
        else:
            api_url = f"https://www.googleapis.com/pagespeedonline/v5/runPagespeed?url={self.url}&key={self.api_key}&strategy=mobile"
            try:
                res = requests.get(api_url, timeout=60)
                res.raise_for_status()
                data = res.json()
                if 'lighthouseResult' in data:
                    lcp = data['lighthouseResult']['audits']['largest-contentful-paint']['displayValue']
                    cls = data['lighthouseResult']['audits']['cumulative-layout-shift']['displayValue']
                    lcp_val = float(re.sub(r'[^\d.]', '', lcp))
                    cls_val = float(re.sub(r'[^\d.]', '', cls))
                    status = 'ok' if lcp_val < 2.5 and cls_val < 0.1 else 'error'
                    result = f"LCP (móvil): {lcp}, CLS (móvil): {cls}."
                else:
                    raise ValueError("Respuesta de API inválida.")
            except Exception as e:
                result = f"No se pudieron obtener los datos. Error: {e}"
                status = 'error'
            
            pain_point = "Una mala puntuación en Core Web Vitals (velocidad, estabilidad) frustra a los usuarios, aumenta la tasa de rebote y es penalizado por Google."
            
        self.pdf.check_item("Core Web Vitals (Velocidad en Móvil)", result, status, pain_point, self.solutions['core_web_vitals'])

    def check_social_links(self):
        socials = {'facebook', 'twitter', 'instagram', 'linkedin', 'youtube'}
        found_links = {s for s in socials if self.soup.find('a', href=re.compile(f"{s}\\.com"))}
        if found_links:
            status = 'ok'
            result = f"Se encontraron enlaces a: {', '.join(found_links).title()}."
        else:
            status = 'warning'
            result = "No se encontraron enlaces a perfiles sociales."
        self.pdf.check_item("Enlaces a Redes Sociales", result, status,
                            "La ausencia de perfiles sociales en la web puede generar desconfianza y priva al negocio de canales de comunicación y tráfico adicionales.",
                            self.solutions['social_links'])

    def check_favicon(self):
        has_favicon = self.soup.find('link', rel=re.compile(r'icon')) is not None
        status = 'ok' if has_favicon else 'error'
        self.pdf.check_item("Favicon", "Se encontró un favicon." if has_favicon else "No se encontró un favicon.", status,
                            "La falta de favicon hace que la web parezca poco profesional en las pestañas del navegador y en los marcadores, afectando la imagen de marca.",
                            self.solutions['favicon'])

    def check_robots_txt(self):
        robots_url = urljoin(self.url, '/robots.txt')
        try:
            res = requests.get(robots_url, timeout=10)
            if res.status_code == 200:
                status = 'ok'
                result = "El archivo robots.txt existe y es accesible."
            else:
                status = 'error'
                result = f"No se encontró el archivo robots.txt (Error {res.status_code})."
        except requests.RequestException:
            status = 'error'
            result = "No se pudo acceder al archivo robots.txt."
        self.pdf.check_item("Archivo robots.txt", result, status,
                            "Un archivo robots.txt ausente o mal configurado puede impedir que Google rastree páginas importantes o, peor aún, que indexe contenido privado.",
                            self.solutions['robots_txt'])

    def check_sitemap(self):
        sitemap_url = urljoin(self.url, '/sitemap.xml')
        try:
            res = requests.head(sitemap_url, timeout=10)
            status = 'ok' if res.status_code == 200 else 'warning'
            result = f"Se encontró un sitemap.xml en la ruta estándar." if status == 'ok' else "No se encontró sitemap.xml en la ruta estándar."
        except requests.RequestException:
            status = 'error'
            result = "No se pudo acceder al sitemap.xml."
        self.pdf.check_item("Sitemap XML", result, status,
                            "Sin un sitemap, Google puede tardar mucho más en descubrir y indexar todas las páginas importantes de la web, especialmente las nuevas.",
                            self.solutions['sitemap'])
    
    def check_domain_age(self):
        try:
            domain_info = whois.whois(self.domain)
            if domain_info.creation_date:
                creation_date = domain_info.creation_date[0] if isinstance(domain_info.creation_date, list) else domain_info.creation_date
                age = (datetime.now() - creation_date).days / 365.25
                status = 'info'
                result = f"Dominio registrado en {creation_date.strftime('%Y-%m-%d')} ({age:.1f} años de antigüedad)."
            else:
                raise ValueError("No se encontró fecha de creación.")
        except Exception:
            status = 'warning'
            result = "No se pudo obtener la información de antigüedad del dominio."
        self.pdf.check_item("Antigüedad y Autoridad del Dominio", result, status,
                            "La autoridad de un dominio es clave para el posicionamiento. Un dominio nuevo o con mala reputación lo tiene más difícil para competir.",
                            self.solutions['domain_age'])
    
    def add_next_steps(self):
        self.pdf.set_font('Arial', '', 10)
        formatted_text = self.proposal_text.format(nombre_agencia=self.config['NOMBRE_AGENCIA'])
        safe_text = formatted_text.encode('latin-1', 'replace').decode('latin-1')
        self.pdf.multi_cell(0, 5, safe_text)

# -------------------------------------------------------------------
#                           EJECUCIÓN DEL SCRIPT
# -------------------------------------------------------------------
if __name__ == "__main__":
    target_url = CONFIGURACION_AGENCIA['URL_A_AUDITAR']
    if not validators.url(target_url):
        print(f"ERROR: La URL '{target_url}' no es válida. Por favor, revísala en la sección de configuración.")
    elif "[EL NOMBRE DE TU AGENCIA]" in CONFIGURACION_AGENCIA['NOMBRE_AGENCIA']:
         print("ERROR: Por favor, rellena el 'NOMBRE_AGENCIA' en la sección de configuración antes de ejecutar.")
    else:
        auditor = HomepageAuditor(CONFIGURACION_AGENCIA, SOLUCIONES_PROPUESTAS, TEXTO_PROPUESTA_FINAL)
        auditor.run()

